#roloFlash 2, v07+

! ***************************************************************************
! *
! *  Sample script for Microchip/Atmel-aWire-Controllers
! *
! *  Task: Read from flash memory via aWire
! *
! *  Copyright (C) 2009-2025 by halec embedded solutions
! *
! ***************************************************************************


! For all files on the microSD card, the following applies:
!  - File name has to be in 8.3 format
!  - File name must contain only CAPITAL LETTERS
!  - (see manual, chapter "Files")


! *** Please adapt to your requirements! ***


targetName = "AT32UC3L064"  ! Take exact name from manual, chapter
                            ! "Specifications"
flashFile  = "FLASHRB.HEX"  ! Specify "", if not needed
eepromFile = ""             ! Specify "", if not needed
busSpeed   = 100000         ! Bus speed in Hertz. 100 kHz are a conservative
                            ! choice, to ensure reliable operation at a low
                            ! target CPU clock or low roloFlash CPU clock. If
                            ! it is known that the target CPU clock is higher,
                            ! the bus speed can be increased appropriately.
roloFlashHiSpeedMode = 0    ! 0 (false): ca. 20mA@3.3V, is slower
                            ! 1 (true):  ca. 80mA@3.3V, is faster

! Green running light from LED 4 to LED 1 -> symbolizes script processing
! (Data transfer direction: read)
! (LED 5 is kept free for display of "Done")
led_runningLight 4, 1, COLOR_GREEN, 200


! ---- Preparations ----
! Delete old log file, if present
f = "LOG.TXT"
if fs_fileExists(0,f)
  fs_remove 0, f
endif


! Write software version of roloFlash and script name to LOG.TXT
print "softwareVersion=", sys_softwareVersion, "\r\n"
print "Running script copied from scripts/Microchip_Atmel/AVR32/aWire/read/ ...\r\n"


! If roloFlashHiSpeedMode has been selected, set CPU clock of roloFlash to
! the maximum
if roloFlashHiSpeedMode
  sys_setCpuClock CPU_CLOCKMAX
endif


! ---- Access to roloFlash's internal target database ----
dbHandle = db_getHandle(targetName)


! ---- Scan aWire bus ----
print "Scanning aWire bus ...\r\n"
busHandle = bus_open(AWIRE, 0, busSpeed)


! ---- Access to target ----
print "Connecting to target ...\r\n"
family       = db_get(dbHandle, DB_FAMILY)
targetHandle = target_open(busHandle, 0, family)
target_setMode targetHandle, PROGRAMMODE


! ---- Check signature / device ID ----
print "Checking signature / device ID ...\r\n"
expectedSignature = db_get(dbHandle, db_coreIDCODE)
signature = target_getDeviceId(targetHandle)
signature = (signature[3] + $100 * signature[2] + $10000 * signature[1] + $1000000 * signature[0]) and $0fffffff
if signature <> expectedSignature
  print "ERROR: Wrong controller detected\r\n"
  ! Abort
  throw USEREXCEPTION + 2
endif


! ---- Get target memory parameters from target database ----
print "Getting target memory parameters from database ...\r\n"
flashSize      = db_get(dbHandle, DB_FLASHSIZE)
flashPageSize  = db_get(dbHandle, DB_FLASHPAGESIZE)
flashOffset = $80000000
print "  Target flash  size     [bytes]: ", flashSize,      "\r\n"
print "  Target flash  pagesize [bytes]: ", flashPageSize,  "\r\n"
print "  Target flash  offset   [bytes]: ", flashOffset,    "\r\n"
target_setMemoryMap targetHandle, FLASH,  MEM_SIZE,     flashSize
target_setMemoryMap targetHandle, FLASH,  MEM_PAGESIZE, flashPageSize
target_setMemoryMap targetHandle, FLASH,  MEM_OFFSET,   flashOffset


! ---- Read from target flash ----
if flashFile <> ""
  print "Reading flash to file ", flashFile, " ...\r\n"
  target_readToFile targetHandle, 0, flashFile, HEX, FLASH, 0, flashSize
else
  print "Flash reading skipped (no flash file specified)\r\n"
endif


! ---- Read from target EEPROM ----
if eepromFile <> ""
  print "Reading EEPROM to file ", eepromFile, " ...\r\n"
  target_readToFile targetHandle, 0, eepromFile, HEX, EEPROM, 0, eepromSize
else
  print "EEPROM reading skipped (no EEPROM file specified)\r\n"
endif


! ---- Postprocessing ----
target_close targetHandle
bus_close busHandle


! ---- Check for possibly occurred exceptions, write           ----
! ---- evaluation to log file and signal it via LEDs           ----
catch exception
print "Duration [ms]: ", sys_getSystemTime(), "\r\n"
catch dummyException  ! If the last print throws an exception
if exception <> 0
  ! There has been an error, record the error in  LOG.TXT
  print "ERROR: Exception ", exception
  ! Throw exception again, after it has been caught. As a result, the number
  ! of the exception gets displayed via LED blink codes. The blink codes
  ! are documented in the manual, chapter "Meaning of LED Codes", subchapter
  ! "Exception has Occurred"
  throw exception
else
  ! No errors: write to log file and switch LED 5 to green
  print "Script ran successfully.\r\n"
  led_on 5, COLOR_GREEN
endif
